<?php
declare(strict_types=1);

class OkxClient {
  private string $apiKey;
  private string $secretKey;
  private string $passphrase;
  private string $base = 'https://www.okx.com';

  public function __construct(string $apiKey, string $secretKey, string $passphrase) {
    $this->apiKey = $apiKey;
    $this->secretKey = $secretKey;
    $this->passphrase = $passphrase;
  }

  private function sign(string $timestamp, string $method, string $pathWithQuery, string $body): string {
    $prehash = $timestamp . strtoupper($method) . $pathWithQuery . $body;
    return base64_encode(hash_hmac('sha256', $prehash, $this->secretKey, true));
  }

  private function request(string $method, string $path, array $query = [], ?array $bodyArr = null, bool $auth = false): array {
    $qs = !empty($query) ? ('?' . http_build_query($query)) : '';
    $url = $this->base . $path . $qs;
    $body = $bodyArr ? json_encode($bodyArr, JSON_UNESCAPED_SLASHES) : '';
    $headers = ['Content-Type: application/json', 'Accept: application/json'];

    if ($auth) {
      $ts = gmdate('Y-m-d\TH:i:s.000\Z');
      $sign = $this->sign($ts, $method, $path . $qs, $body);
      $headers[] = 'OK-ACCESS-KEY: ' . $this->apiKey;
      $headers[] = 'OK-ACCESS-SIGN: ' . $sign;
      $headers[] = 'OK-ACCESS-TIMESTAMP: ' . $ts;
      $headers[] = 'OK-ACCESS-PASSPHRASE: ' . $this->passphrase;
    }

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, strtoupper($method));
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    if (in_array(strtoupper($method), ['POST','PUT','DELETE'], true)) curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
    $resp = curl_exec($ch);
    $err = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) throw new RuntimeException("cURL error: " . $err);
    $json = json_decode($resp, true);
    if (!is_array($json)) throw new RuntimeException("Bad JSON (HTTP $code): " . substr($resp, 0, 200));
    return $json;
  }

  public function ticker(string $instId): array {
    return $this->request('GET', '/api/v5/market/ticker', ['instId' => $instId]);
  }

  public function balance(string $ccy = 'USDT'): array {
    return $this->request('GET', '/api/v5/account/balance', ['ccy' => $ccy], null, true);
  }

  public function placeSpotBuy(string $instId, string $quoteSz): array {
    $body = [
      'instId' => $instId,
      'tdMode' => 'cash',
      'side' => 'buy',
      'ordType' => 'market',
      'tgtCcy' => 'base_ccy',
      'quoteSz' => $quoteSz,
    ];
    return $this->request('POST', '/api/v5/trade/order', [], $body, true);
  }
}
