<?php
declare(strict_types=1);

if (php_sapi_name() !== 'cli') {
  if (session_status() !== PHP_SESSION_ACTIVE) @session_start();
}

function cfg(): array {
  $local = __DIR__ . '/../config/config.local.php';
  if (!file_exists($local)) return require __DIR__ . '/../config/config.sample.php';
  return require $local;
}

function db(): PDO {
  static $pdo = null;
  if ($pdo instanceof PDO) return $pdo;
  $c = cfg();
  $dsn = "mysql:host={$c['db']['host']};dbname={$c['db']['name']};charset=utf8mb4";
  $pdo = new PDO($dsn, $c['db']['user'], $c['db']['pass'], [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}


function ensure_schema_migrations(PDO $pdo): void {
  // v0.3 adds settings.min_usdt_balance
  try {
    $pdo->query("SELECT min_usdt_balance FROM settings LIMIT 1");
  } catch (Throwable $e) {
    try { $pdo->exec("ALTER TABLE settings ADD COLUMN min_usdt_balance FLOAT NOT NULL DEFAULT 5.5"); } catch (Throwable $e2) {}
  }
}


function log_file(string $m): void {
  @file_put_contents(__DIR__ . '/../logs/engine.log', "[".date('c')."] ".$m."\n", FILE_APPEND);
}

function log_db(string $m): void {
  try {
    $pdo = db();
    ensure_schema_migrations($pdo);
    $pdo->prepare("INSERT INTO logs(message, created_at) VALUES(?, NOW())")->execute([$m]);
  } catch (Throwable $e) {
    log_file("DBLOG_FAIL ".$e->getMessage()." | ".$m);
  }
}

function app_key(): string {
  $c = cfg();
  if (empty($c['app_key'])) throw new RuntimeException("APP_KEY not set. Run install.php");
  return $c['app_key'];
}

function enc_str(string $p): string {
  $key = hash('sha256', app_key(), true);
  $iv = random_bytes(16);
  $c = openssl_encrypt($p, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
  return base64_encode($iv.$c);
}

function dec_str(string $b64): string {
  $key = hash('sha256', app_key(), true);
  $raw = base64_decode($b64, true);
  if ($raw === false || strlen($raw) < 17) throw new RuntimeException("Bad ciphertext");
  $iv = substr($raw, 0, 16);
  $c = substr($raw, 16);
  $p = openssl_decrypt($c, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
  if ($p === false) throw new RuntimeException("Decrypt failed");
  return $p;
}

function is_post(): bool { return ($_SERVER['REQUEST_METHOD'] ?? '') === 'POST'; }
function h(?string $s): string { return htmlspecialchars($s ?? '', ENT_QUOTES|ENT_SUBSTITUTE, 'UTF-8'); }

function csrf_token(): string {
  if (php_sapi_name() === 'cli') return '';
  if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
  return (string)$_SESSION['csrf_token'];
}
function csrf_check(string $t): void {
  if (php_sapi_name() === 'cli') return;
  if (empty($_SESSION['csrf_token']) || !hash_equals((string)$_SESSION['csrf_token'], $t)) {
    throw new RuntimeException("Invalid CSRF token. Refresh and try again.");
  }
}
