<?php
declare(strict_types=1);
require_once __DIR__ . '/../core/bootstrap.php';
require_once __DIR__ . '/../core/OkxClient.php';

function get_settings(): array {
  $pdo = db();
  ensure_schema_migrations($pdo);
  $row = $pdo->query("SELECT * FROM settings ORDER BY id ASC LIMIT 1")->fetch();
  if (!$row) throw new RuntimeException("Settings missing. Run install.php");
  $row['allowed_pairs'] = array_values(array_filter(array_map('trim', explode(',', (string)$row['allowed_pairs']))));
  return $row;
}

function get_secrets(): array {
  $row = db()->query("SELECT * FROM secrets ORDER BY id DESC LIMIT 1")->fetch();
  if (!$row) throw new RuntimeException("Secrets missing. Run install.php");
  return [
    'api_key' => dec_str($row['okx_api_key']),
    'secret_key' => dec_str($row['okx_secret_key']),
    'passphrase' => dec_str($row['okx_passphrase']),
  ];
}

function today_executed_count(): int {
  $r = db()->query("SELECT COUNT(*) c FROM trades WHERE status='executed' AND DATE(created_at)=CURDATE()")->fetch();
  return (int)($r['c'] ?? 0);
}

function pending_proposals_count(): int {
  $r = db()->query("SELECT COUNT(*) c FROM trades WHERE status='proposal'")->fetch();
  return (int)($r['c'] ?? 0);
}

function open_positions_count(): int {
  $r = db()->query("SELECT COUNT(*) c FROM positions WHERE status='open'")->fetch();
  return (int)($r['c'] ?? 0);
}

function okx_usdt_available(): float {
  $secrets = get_secrets();
  $okx = new OkxClient($secrets['api_key'], $secrets['secret_key'], $secrets['passphrase']);
  $bal = $okx->balance('USDT');
  $avail = 0.0;
  if (!empty($bal['data'][0]['details'])) {
    foreach ($bal['data'][0]['details'] as $d) {
      if (($d['ccy'] ?? '') === 'USDT') { $avail = (float)($d['availBal'] ?? 0); break; }
    }
  }
  return $avail;
}

function simple_signal(string $instId): ?array {
  $secrets = get_secrets();
  $okx = new OkxClient($secrets['api_key'], $secrets['secret_key'], $secrets['passphrase']);
  $tick = $okx->ticker($instId);
  $last = (float)($tick['data'][0]['last'] ?? 0);
  if ($last <= 0) return null;
  $digit = ((int)round($last)) % 10;
  if ($digit <= 2) return null;
  return [
    'pair' => $instId,
    'action' => 'buy',
    'price' => $last,
    'reason' => 'Starter demo signal (will be replaced with full strategy)',
  ];
}

function create_trade_record(array $sig, string $status): int {
  $stmt = db()->prepare("INSERT INTO trades(pair, action, qty, price, status, reason) VALUES (?,?,?,?,?,?)");
  $stmt->execute([$sig['pair'], $sig['action'], 0, $sig['price'], $status, $sig['reason'] ?? null]);
  return (int)db()->lastInsertId();
}

function run_engine(): void {
  $s = get_settings();

  if ((int)$s['kill_switch'] === 1) { log_file("KILL_SWITCH active. Engine stopped."); return; }
  if (pending_proposals_count() > 0) { log_file("Pending proposal exists. Waiting for approve/reject."); return; }

  if (today_executed_count() >= (int)$s['max_trades_per_day']) { log_file("Max executed trades/day reached."); return; }
  if (open_positions_count() >= (int)$s['max_open_positions']) { log_file("Max open positions reached."); return; }

  $minBal = (float)($s['min_usdt_balance'] ?? 5.5);
  $avail = okx_usdt_available();
  if ($avail < $minBal) { log_file("Low USDT balance (avail={$avail}). Need at least {$minBal} to create proposals."); return; }

  foreach ($s['allowed_pairs'] as $pair) {
    $sig = simple_signal($pair);
    if (!$sig) continue;

    $id = create_trade_record($sig, 'proposal');
    log_file("Created manual proposal #$id {$pair} @ {$sig['price']}");
    log_db("Proposal created #$id {$pair}");
    return;
  }

  log_file("No signals found this run.");
}
